function [u,ll] = laplaceneufdsor( xsw, xne, u0, cc, alpha,  ...
                                  tolerance, maxiter, ncheck)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
%  [u,ll] = laplacedirfdsor( xso, xne, u0, alpha, tolerance, ...          %
%                                maxiter, ncheck)                         %
%                                                                         %
%  Resuelve la ecuacion de laplace en un dominio rectangular mediante     %
%  diferencias finitas. Las ecuaciones de las diferencias finitas se      %
%  resuelven por sobre-relajacion sucesiva (SOR). Supone condiciones      %
%  de contorno de Neumann y malla equiespaciada. Por sencillez,           %
%  se supone que el valor de las derivadas es constante a lo largo de     %
%  cada contorno.                                                         %
%                                                                         %
%  ENTRADA                                                                %
%     xso   coordenadas del punto inferior izquierdo (suroeste). Array    %
%           de dos valores con las coordenadas xy del punto.              %
%     xne   idem para el punto superior derecho.                          %
%      nx   numero de puntos de la malla en las direcciones XY. Array     %
%           de dos valores.                                               %
%      u0   estimacion inicial de la solucion. Incluye tambien los        %
%           valores en los contornos, que no van a ser alterados.         %
%      cc   valores de las derivadas en los contornos en un array de      %
%           cuatro elementos en orden N, S, E, W. La derivada se supone   %
%           la derivada normal                                            %
%                                                                         %
%tolerance  tolerancia requerida (variacion relativa entre iteraciones    %
%           sucesivas                                                     %
%  maxiter  maximo numero de iteraciones permitido                        %
%  ncheck   numero de iteraciones tras las cuales se desea un 'informe'   %
%                                                                         %
%  SALIDA                                                                 %
%       u  solucion encontrada. Array de nx filas y ny columnas.          %
%      ll  numero de iteraciones realizadas                               %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% resolvemos el problema en una malla aumentada
nx0 = size(u0);
u  = zeros( nx0 +2 );
u( 2:end-1, 2:end-1 ) = u0;

nx = size(u);
uu = zeros(nx);
dx = xne - xsw;
dx = dx./(nx-1);
epsilon =  0.5 / (dx(1)^2+dx(2)^2) * [ dx(2)^2 dx(1)^2] ;
% condiciones de contorno
ccn = cc(1);
ccs = cc(2);
cce = cc(3);
ccw = cc(4);

done = false; % parametro de control del bucle
ll = 0;       % contador de iteraciones

kk = 0;       % contador auxiliara para visualizacion
epskk = ones(1,maxiter/ncheck)*tolerance;

figure(9); clf;

while ~done
    
    done = true;
    ll = ll + 1;
    if( ll > maxiter ), 
        disp('Demasiadas iteraciones');
        u = 0.0;
        break;
    end;
    
    uu = u;
    % asignamos valores a nodos virtuales
    % LADOS (evitamos esquinas)
    %
    %   lado S (primera fila, columna corriendo)
    u( 1, 2:end-1 )  = u(      3, 2:end-1) - 2.0*dx(1) * ccn;
    %   lado N (ultima fila, columna corriendo)
    u(end, 2:end-1)  = u(  end-2, 2:end-1) - 2.0*dx(1) * ccs;
    %   lado E (primera columna, fila corriendo)
    u( 2:end-1, 1)   = u( 2:end-1, 3     ) - 2.0*dx(2) * ccw;
    %   lado W (ultima columna, fila corriendo)
    u( 2:end-1, end) = u( 2:end-1,  end-2) - 2.0*dx(2) * cce; 
    % vecinos de las esquinas
    
    % puntos interiores
    % Bucle en u. . (Gauss-Seidel)
    for ii = 2:nx(1)-1
        for jj=2:nx(2)-1
            u(ii,jj) = epsilon(1)*( u(ii+1, jj) + u(ii-1,jj) ) + ...
                       epsilon(2)*( u(ii, jj+1) + u(ii, jj-1) );
        end
    end
    % SOR
    u = ( 1.0 + alpha)*u - alpha*uu;
    
    % calculamos la maxima diferencia entre una iteracion y la
    % siguiente y la almacenamos en epsjj
    
    epsll =  sum(sum(abs(uu-u))) / sum(sum(abs(uu)));
    if ( epsll > tolerance )
        done = false;
    end
    
    
    % cada ncheck iteraciones guardamos el error relativo y dibujamos
    % la solucion (la funcion rem devuelve el resto de la division).
    if( rem( ll-1 , ncheck ) == 0)
        surf(u(2:end-1,2:end-1));
        view(26, 40);
        legend(sprintf('iteracion %d\n eps %e',ll,epsll));
        pause(0.2);
        kk = kk+1;
        epskk(kk) = epsll;
    end
end
fprintf('\n iteraciones = %d\n',ll);
% recuperamos los puntos de la malla originales
u = u(2:end-1,2:end-1);

figure(10); clf;
semilogy( 1:ncheck:(kk*ncheck), epskk(1:kk),'o');
xlabel('numero de iteracion'); ylabel('error relativo');



